// 在淘宝/天猫详情页收集图片链接
(function () {
  // 便于 UI 按钮或 popup 触发
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    if (msg.type === "COLLECT_AND_DOWNLOAD") {
      collectAndSend();
      sendResponse({ ok: true });
    }
  });

  // 可在页面注入一个轻量按钮（可选）
  injectQuickButton();

  async function collectAndSend() {
    const title = getTitleSafe();
    const prefix = sanitizeFilename(title) || "taobao-product";

    const urls = [
      ...collectMainGallery(),
      ...collectSkuImages(),
      ...await collectDescImages()
    ];

    chrome.runtime.sendMessage({
      type: "DOWNLOAD_IMAGES",
      payload: { images: urls, prefix }
    }, (res) => {
      if (res?.ok) {
        toast(`开始下载，共 ${res.count} 张`);
      } else {
        toast(`下载失败：${res?.error || "未知错误"}`);
      }
    });
  }

  function collectMainGallery() {
    const urls = new Set();

    // 淘宝/天猫 PC 主图区域选择器尝试
    const candidates = [
      ".tb-thumb img",              // 淘宝缩略
      ".tb-pic img",                // 淘宝主图
      ".tb-gallery img",            // 淘宝/天猫主图容器
      ".viewer-container img",      // 新版
      ".swiper-slide img",          // 移动端
      "img[data-src]",
      "img[src]"
    ];

    candidates.forEach(sel => {
      document.querySelectorAll(sel).forEach(img => {
        const u = pickBestSrc(img);
        if (isProductImage(u)) urls.add(u);
      });
    });

    return Array.from(urls);
  }

  function collectSkuImages() {
    const urls = new Set();
    // SKU 区域常见选择器
    const candidates = [
      ".tb-sku .sku img",
      ".J_Prop .J_TSaleProp li img",
      ".tm-sale-prop img",
      ".sku-selector img"
    ];

    candidates.forEach(sel => {
      document.querySelectorAll(sel).forEach(img => {
        const u = pickBestSrc(img);
        if (isProductImage(u)) urls.add(u);
      });
    });

    return Array.from(urls);
  }

  async function collectDescImages() {
    const urls = new Set();

    // 描述区域可能是 iframe（天猫/淘宝常见）
    // 常见 id：J_DivItemDesc, descIframe, J_Desc, J_ItemDesc
    const iframe = document.querySelector(
      "iframe#descIframe, iframe#J_Desc, iframe#J_ItemDesc, iframe#J_DivItemDesc"
    );

    if (iframe && iframe.contentDocument) {
      iframe.contentDocument.querySelectorAll("img").forEach(img => {
        const u = pickBestSrc(img);
        if (isProductImage(u)) urls.add(u);
      });
    } else {
      // 有些新版直接内嵌在 div
      const descCandidates = [
        "#J_DivItemDesc",
        "#description",
        ".item-desc",
        ".J_ItemDesc",
        ".tm-description",
        ".pc-detail-content",
        ".content"
      ];
      descCandidates.forEach(sel => {
        document.querySelectorAll(`${sel} img`).forEach(img => {
          const u = pickBestSrc(img);
          if (isProductImage(u)) urls.add(u);
        });
      });
    }

    // 兜底：全页图片（可能包含非产品图，稍后过滤）
    document.querySelectorAll("img").forEach(img => {
      const u = pickBestSrc(img);
      if (isProductImage(u)) urls.add(u);
    });

    return Array.from(urls);
  }

  function pickBestSrc(img) {
    // 优先取原图属性
    const attrs = ["data-src", "data-ks-lazyload", "data-lazyload", "data-original", "srcset", "src"];
    for (const a of attrs) {
      const v = img.getAttribute(a);
      if (v && v.trim()) {
        if (a === "srcset") {
          // 取最大分辨率
          const last = v.split(",").map(s => s.trim()).pop();
          if (last) return (last.split(" ")[0] || "").trim();
        } else {
          return v.trim();
        }
      }
    }
    return "";
  }

  function isProductImage(url) {
    if (!url) return false;
    const u = url.toLowerCase();
    // 常见域：alicdn、taobaocdn
    if (u.includes("alicdn.com") || u.includes("taobaocdn.com") || u.includes("tbcdn.cn")) return true;
    // 详情可能自托管，也放行常见图片后缀
    return /\.(jpg|jpeg|png|webp)(\?.*)?$/.test(u);
  }

  function getTitleSafe() {
    // 标题选择器兜底
    const candidates = [
      "#J_Title h3",
      ".tb-detail-hd h1",
      ".tb-main-title",
      ".tm-detail-title",
      "title"
    ];
    for (const sel of candidates) {
      const el = sel === "title" ? document.querySelector("title") : document.querySelector(sel);
      if (el && el.textContent) {
        return el.textContent.trim();
      }
    }
    return "taobao-product";
  }

  function sanitizeFilename(name) {
    return name.replace(/[\\/:*?"<>|]/g, "").slice(0, 60).trim();
  }

  function injectQuickButton() {
    const btn = document.createElement("button");
    btn.textContent = "下载产品图";
    btn.style.position = "fixed";
    btn.style.right = "16px";
    btn.style.bottom = "16px";
    btn.style.zIndex = "999999";
    btn.style.padding = "10px 14px";
    btn.style.background = "#ff5000";
    btn.style.color = "#fff";
    btn.style.border = "none";
    btn.style.borderRadius = "4px";
    btn.style.cursor = "pointer";
    btn.style.boxShadow = "0 2px 8px rgba(0,0,0,0.25)";
    btn.addEventListener("click", collectAndSend);
    document.body.appendChild(btn);
  }

  function toast(text) {
    const div = document.createElement("div");
    div.textContent = text;
    div.style.position = "fixed";
    div.style.left = "50%";
    div.style.bottom = "70px";
    div.style.transform = "translateX(-50%)";
    div.style.background = "rgba(0,0,0,0.75)";
    div.style.color = "#fff";
    div.style.padding = "8px 12px";
    div.style.borderRadius = "4px";
    div.style.fontSize = "12px";
    div.style.zIndex = "999999";
    document.body.appendChild(div);
    setTimeout(() => div.remove(), 2000);
  }
})();
